/*
 * Decompiled with CFR 0.152.
 */
package commonSense.math.linear;

import commonSense.math.linear.InvalidMatrixException;
import commonSense.math.linear.LUDecomposition;
import commonSense.math.linear.MatrixIndexException;
import commonSense.math.linear.PointXD;
import commonSense.math.linear.QRDecomposition;
import commonSense.math.linear.SingularValueDecomposition;
import commonSense.math.util.MathUtils;
import java.awt.geom.Point2D;
import java.io.Serializable;

public class Matrix
implements Serializable {
    static final long serialVersionUID = 4237564493130426188L;
    protected double[][] data = null;
    protected int rows = 0;
    protected int columns = 0;
    protected LUDecomposition lu = null;
    private int[] permutation = null;
    private int parity = 1;
    protected static double TOO_SMALL = 1.0E-11;

    public Matrix() {
        this.lu = null;
    }

    public Matrix(int rowDimension, int columnDimension) {
        this.rows = rowDimension;
        this.columns = columnDimension;
        this.data = new double[rowDimension][columnDimension];
        this.lu = null;
    }

    public Matrix(int rowDimension, int columnDimension, double value) {
        this.rows = rowDimension;
        this.columns = columnDimension;
        this.data = new double[rowDimension][columnDimension];
        for (int x = 0; x < this.rows; ++x) {
            for (int y = 0; y < this.columns; ++y) {
                this.data[x][y] = value;
            }
        }
        this.lu = null;
    }

    public Matrix(double[][] d) {
        this.copyIn(d);
        this.lu = null;
    }

    public Matrix(Point2D.Double[] d) {
        this.rows = d.length;
        this.columns = 2;
        this.data = new double[this.rows][this.columns];
        for (int x = 0; x < this.rows; ++x) {
            this.data[x][0] = d[x].getX();
            this.data[x][1] = d[x].getY();
        }
        this.lu = null;
    }

    public Matrix(PointXD[] d) {
        this.rows = d.length;
        this.columns = d[0].dim;
        this.data = new double[this.rows][this.columns];
        for (int x = 0; x < this.rows; ++x) {
            for (int y = 0; y < this.columns; ++y) {
                this.data[x][y] = d[x].getCoordinate(y);
            }
        }
        this.lu = null;
    }

    public Matrix(double[] v) {
        this.rows = v.length;
        this.columns = 1;
        this.data = new double[this.rows][1];
        for (int row = 0; row < this.rows; ++row) {
            this.data[row][0] = v[row];
        }
        this.lu = null;
    }

    public void clear() {
        this.data = null;
        this.lu = null;
        this.rows = 0;
        this.columns = 0;
    }

    public Matrix copy() {
        return new Matrix(this.copyOut());
    }

    public Matrix add(Matrix m) throws IllegalArgumentException {
        if (this.columns != m.columns || this.rows != m.rows) {
            throw new IllegalArgumentException("Matrix dimension mismatch");
        }
        double[][] outData = new double[this.rows][this.columns];
        double[][] mData = m.getData();
        for (int row = 0; row < this.rows; ++row) {
            for (int col = 0; col < this.columns; ++col) {
                outData[row][col] = this.data[row][col] + mData[row][col];
            }
        }
        return new Matrix(outData);
    }

    public Matrix subtract(Matrix m) throws IllegalArgumentException {
        if (this.columns != m.columns || this.rows != m.rows) {
            throw new IllegalArgumentException("matrix dimension mismatch");
        }
        double[][] outData = new double[this.rows][this.columns];
        double[][] mData = m.getData();
        for (int row = 0; row < this.rows; ++row) {
            for (int col = 0; col < this.columns; ++col) {
                outData[row][col] = this.data[row][col] - mData[row][col];
            }
        }
        return new Matrix(outData);
    }

    public Matrix scalarAdd(double d) {
        double[][] outData = new double[this.rows][this.columns];
        for (int row = 0; row < this.rows; ++row) {
            for (int col = 0; col < this.columns; ++col) {
                outData[row][col] = this.data[row][col] + d;
            }
        }
        return new Matrix(outData);
    }

    public Matrix scalarMultiply(double d) {
        double[][] outData = new double[this.rows][this.columns];
        for (int row = 0; row < this.rows; ++row) {
            for (int col = 0; col < this.columns; ++col) {
                outData[row][col] = this.data[row][col] * d;
            }
        }
        return new Matrix(outData);
    }

    public Matrix multiply(Matrix m) throws IllegalArgumentException {
        if (this.columns != m.rows) {
            throw new IllegalArgumentException("Matrices are not multiplication compatible (" + this.columns + " columns expected, " + m.rows + " found.");
        }
        double[][] mData = m.getData();
        double[][] outData = new double[this.rows][m.columns];
        double sum = 0.0;
        for (int row = 0; row < this.rows; ++row) {
            for (int col = 0; col < m.columns; ++col) {
                sum = 0.0;
                for (int i = 0; i < this.columns; ++i) {
                    sum += this.data[row][i] * mData[i][col];
                }
                outData[row][col] = sum;
            }
        }
        return new Matrix(outData);
    }

    public Matrix preMultiply(Matrix m) throws IllegalArgumentException {
        return m.multiply(this);
    }

    public Matrix kroneckerProduct(Matrix m) {
        double[][] product = new double[this.rows * m.rows][this.columns * m.columns];
        double[][] mData = m.getData();
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.columns; ++j) {
                for (int k = 0; k < m.rows; ++k) {
                    for (int l = 0; l < m.columns; ++l) {
                        product[i * m.rows + k][j * m.columns + l] = this.data[i][j] * mData[k][l];
                    }
                }
            }
        }
        return new Matrix(product);
    }

    public Matrix kroneckerPreProduct(Matrix m) {
        return m.kroneckerProduct(this);
    }

    public Matrix hadamardProduct(Matrix m) {
        if (this.columns != m.columns || this.rows != m.rows) {
            throw new IllegalArgumentException("Matrices are not multiplication compatible");
        }
        double[][] mData = m.getData();
        double[][] outData = new double[this.rows][this.columns];
        for (int row = 0; row < this.rows; ++row) {
            for (int col = 0; col < this.columns; ++col) {
                outData[row][col] = this.data[row][col] * mData[row][col];
            }
        }
        return new Matrix(outData);
    }

    public Matrix hadamardDivide(Matrix m) {
        if (this.columns != m.columns || this.rows != m.rows) {
            throw new IllegalArgumentException("Matrices are not multiplication compatible");
        }
        double[][] mData = m.getData();
        double[][] outData = new double[this.rows][this.columns];
        for (int row = 0; row < this.rows; ++row) {
            for (int col = 0; col < this.columns; ++col) {
                outData[row][col] = this.data[row][col] / mData[row][col];
            }
        }
        return new Matrix(outData);
    }

    public double[][] getData() {
        return this.copyOut();
    }

    public void setData(double[][] inData) {
        this.copyIn(inData);
        this.lu = null;
    }

    public double[][] getDataRef() {
        return this.data;
    }

    public void setDataRef(double[][] inData) {
        this.data = inData;
        this.lu = null;
    }

    public void insertRow(double[] inData, int position) {
        int x;
        if (this.columns != inData.length) {
            throw new IllegalArgumentException("Length of inData (" + inData.length + ") and number of columns (" + this.columns + ") mismatch!");
        }
        if (position > this.rows) {
            throw new IllegalArgumentException("Position (" + position + ") outside allowable range (0-" + this.rows + ")");
        }
        this.lu = null;
        ++this.rows;
        double[][] dataNew = new double[this.rows][this.columns];
        for (x = 0; x < position; ++x) {
            System.arraycopy(this.data[x], 0, dataNew[x], 0, this.columns);
        }
        System.arraycopy(inData, 0, dataNew[position], 0, this.columns);
        for (x = position + 1; x < this.rows; ++x) {
            System.arraycopy(this.data[x - 1], 0, dataNew[x], 0, this.columns);
        }
        this.copyIn(dataNew);
    }

    public void removeRow(int position) {
        int x;
        if (position > this.rows - 1 || position < 0) {
            throw new IllegalArgumentException("Position (" + position + ") outside allowable range (0-" + (this.rows - 1) + ")");
        }
        this.lu = null;
        --this.rows;
        double[][] dataNew = new double[this.rows][this.columns];
        for (x = 0; x < position; ++x) {
            System.arraycopy(this.data[x], 0, dataNew[x], 0, this.columns);
        }
        for (x = position; x < this.rows; ++x) {
            System.arraycopy(this.data[x + 1], 0, dataNew[x], 0, this.columns);
        }
        this.copyIn(dataNew);
    }

    public double getNorm() {
        double maxColSum = 0.0;
        for (int col = 0; col < this.columns; ++col) {
            double sum = 0.0;
            for (int row = 0; row < this.rows; ++row) {
                sum += StrictMath.abs(this.data[row][col]);
            }
            maxColSum = StrictMath.max(maxColSum, sum);
        }
        return maxColSum;
    }

    public double[] getRow(int row) throws MatrixIndexException {
        if (!this.isValidCoordinate(row, 0)) {
            throw new MatrixIndexException("illegal row argument");
        }
        double[] out = new double[this.columns];
        System.arraycopy(this.data[row], 0, out, 0, this.columns);
        return out;
    }

    public Matrix getRowMatrix(int row) throws MatrixIndexException {
        if (!this.isValidCoordinate(row, 0)) {
            throw new MatrixIndexException("illegal row argument");
        }
        double[][] out = new double[1][this.columns];
        System.arraycopy(this.data[row], 0, out[0], 0, this.columns);
        return new Matrix(out);
    }

    public double[] getColumn(int col) throws MatrixIndexException {
        if (!this.isValidCoordinate(0, col)) {
            throw new MatrixIndexException("illegal column argument");
        }
        double[] out = new double[this.rows];
        for (int row = 0; row < this.rows; ++row) {
            out[row] = this.data[row][col];
        }
        return out;
    }

    public Matrix getColumnMatrix(int col) throws MatrixIndexException {
        if (!this.isValidCoordinate(0, col)) {
            throw new MatrixIndexException("illegal row argument");
        }
        double[][] out = new double[this.rows][1];
        for (int y = 0; y < this.rows; ++y) {
            out[y][0] = this.data[y][col];
        }
        return new Matrix(out);
    }

    public double getEntry(int row, int column) throws MatrixIndexException {
        if (!this.isValidCoordinate(row, column)) {
            throw new MatrixIndexException("matrix entry does not exist");
        }
        return this.data[row][column];
    }

    public void setEntry(int row, int column, double value) throws MatrixIndexException {
        if (!this.isValidCoordinate(row, column)) {
            throw new MatrixIndexException("matrix entry does not exist");
        }
        this.data[row][column] = value;
        this.lu = null;
    }

    public Matrix transpose() {
        Matrix out = new Matrix(this.columns, this.rows);
        double[][] outData = out.getDataRef();
        for (int row = 0; row < this.rows; ++row) {
            for (int col = 0; col < this.columns; ++col) {
                outData[col][row] = this.data[row][col];
            }
        }
        return out;
    }

    public Matrix concatenateColumns(Matrix secondMatrix) {
        if (this.columns != secondMatrix.columns) {
            throw new IllegalArgumentException("Matrices have different number of columns");
        }
        int newRows = this.rows + secondMatrix.rows;
        double[][] out = new double[newRows][this.columns];
        for (int x = 0; x < this.rows; ++x) {
            System.arraycopy(this.data[x], 0, out[x], 0, this.columns);
        }
        double[][] secondData = secondMatrix.getData();
        for (int x = 0; x < secondMatrix.rows; ++x) {
            System.arraycopy(secondData[x], 0, out[x + this.rows], 0, this.columns);
        }
        return new Matrix(out);
    }

    public Matrix inverse() throws InvalidMatrixException {
        return this.solve(Matrix.getIdentity(this.rows));
    }

    public Matrix pseudoInverse() throws InvalidMatrixException {
        return new SingularValueDecomposition(this).inverse();
    }

    public double getDeterminant() throws InvalidMatrixException {
        if (!this.isSquare()) {
            throw new InvalidMatrixException("matrix is not square");
        }
        if (this.lu == null) {
            this.lu = new LUDecomposition(this);
        }
        if (this.lu.isSingular()) {
            return 0.0;
        }
        return this.lu.getDeterminant();
    }

    public boolean isSquare() {
        return this.columns == this.rows;
    }

    public boolean isSingular() {
        if (this.lu == null) {
            this.lu = new LUDecomposition(this);
        }
        return this.lu.isSingular();
    }

    public int[] getDimensions() {
        int[] dim = new int[]{this.rows, this.columns};
        return dim;
    }

    public int getRowDimension() {
        return this.rows;
    }

    public int getColumnDimension() {
        return this.columns;
    }

    public double getTrace() throws IllegalArgumentException {
        if (!this.isSquare()) {
            throw new IllegalArgumentException("matrix is not square");
        }
        double trace = this.data[0][0];
        for (int i = 1; i < this.rows; ++i) {
            trace += this.data[i][i];
        }
        return trace;
    }

    public double[] operate(double[] v) throws IllegalArgumentException {
        if (v.length != this.columns) {
            throw new IllegalArgumentException("vector has wrong length");
        }
        double[] out = new double[v.length];
        for (int row = 0; row < this.rows; ++row) {
            double sum = 0.0;
            for (int i = 0; i < this.columns; ++i) {
                sum += this.data[row][i] * v[i];
            }
            out[row] = sum;
        }
        return out;
    }

    public double[] preMultiply(double[] v) throws IllegalArgumentException {
        if (v.length != this.rows) {
            throw new IllegalArgumentException("vector has wrong length");
        }
        double[] out = new double[this.columns];
        for (int col = 0; col < this.columns; ++col) {
            double sum = 0.0;
            for (int i = 0; i < this.rows; ++i) {
                sum += this.data[i][col] * v[i];
            }
            out[col] = sum;
        }
        return out;
    }

    public double[] solve(double[] b) throws IllegalArgumentException, InvalidMatrixException {
        if (b.length != this.rows) {
            throw new IllegalArgumentException("constant vector has wrong length");
        }
        Matrix bMatrix = new Matrix(b);
        double[][] solution = this.solve(bMatrix).getDataRef();
        double[] out = new double[this.rows];
        for (int row = 0; row < this.rows; ++row) {
            out[row] = solution[row][0];
        }
        return out;
    }

    public Matrix solve(Matrix b) throws IllegalArgumentException, InvalidMatrixException {
        if (b.rows != this.rows) {
            throw new IllegalArgumentException("Incorrect row dimension");
        }
        if (!this.isSquare()) {
            throw new InvalidMatrixException("coefficient matrix is not square");
        }
        return this.rows == this.columns ? new LUDecomposition(this).solve(b) : new QRDecomposition(this).solve(b);
    }

    public boolean equals(Object object) {
        if (object == this) {
            return true;
        }
        if (!(object instanceof Matrix)) {
            return false;
        }
        Matrix m = (Matrix)object;
        int nRows = this.getRowDimension();
        int nCols = this.getColumnDimension();
        if (m.getColumnDimension() != nCols || m.getRowDimension() != nRows) {
            return false;
        }
        for (int row = 0; row < nRows; ++row) {
            for (int col = 0; col < nCols; ++col) {
                if (Double.doubleToLongBits(this.data[row][col]) == Double.doubleToLongBits(m.getEntry(row, col))) continue;
                return false;
            }
        }
        return true;
    }

    public int hashCode() {
        int ret = 7;
        int nRows = this.getRowDimension();
        int nCols = this.getColumnDimension();
        ret = ret * 31 + nRows;
        ret = ret * 31 + nCols;
        for (int row = 0; row < nRows; ++row) {
            for (int col = 0; col < nCols; ++col) {
                ret = ret * 31 + (11 * (row + 1) + 17 * (col + 1)) * MathUtils.hash(this.data[row][col]);
            }
        }
        return ret;
    }

    public double[] rowMeans() {
        double[] means = new double[this.rows];
        for (int x = 0; x < this.rows; ++x) {
            for (int y = 0; y < this.columns; ++y) {
                int n = x;
                means[n] = means[n] + this.data[x][y] / (double)this.columns;
            }
        }
        return means;
    }

    public double[] columnMeans() {
        double[] means = new double[this.columns];
        for (int x = 0; x < this.rows; ++x) {
            for (int y = 0; y < this.columns; ++y) {
                int n = y;
                means[n] = means[n] + this.data[x][y] / (double)this.rows;
            }
        }
        return means;
    }

    public double[] columnVariances(int type) {
        int N = this.rows - type;
        double[] means = this.columnMeans();
        double[] variances = new double[this.columns];
        for (int x = 0; x < this.rows; ++x) {
            int y = 0;
            while (y < this.columns) {
                double temp = this.data[x][y] - means[y];
                temp *= temp;
                int n = y++;
                variances[n] = variances[n] + temp / (double)N;
            }
        }
        return variances;
    }

    public double[] columnSDs(int type) {
        double[] variances = this.columnVariances(type);
        double[] SDs = new double[variances.length];
        for (int x = 0; x < variances.length; ++x) {
            SDs[x] = StrictMath.sqrt(variances[x]);
        }
        return SDs;
    }

    public Matrix getSubMatrix(int startRow, int endRow, int startColumn, int endColumn) throws MatrixIndexException {
        Matrix subMatrix = new Matrix(endRow - startRow + 1, endColumn - startColumn + 1);
        double[][] subMatrixData = subMatrix.getDataRef();
        try {
            for (int i = startRow; i <= endRow; ++i) {
                for (int j = startColumn; j <= endColumn; ++j) {
                    subMatrixData[i - startRow][j - startColumn] = this.data[i][j];
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new MatrixIndexException("matrix dimension mismatch");
        }
        return subMatrix;
    }

    public Matrix getSubMatrix(int[] selectedRows, int[] selectedColumns) throws MatrixIndexException {
        Matrix subMatrix = new Matrix(selectedRows.length, selectedColumns.length);
        double[][] subMatrixData = subMatrix.getDataRef();
        try {
            for (int i = 0; i < selectedRows.length; ++i) {
                for (int j = 0; j < selectedColumns.length; ++j) {
                    subMatrixData[i][j] = this.data[selectedRows[i]][selectedColumns[j]];
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new MatrixIndexException("matrix dimension mismatch");
        }
        return subMatrix;
    }

    public Matrix getAbsolute() {
        double[][] newData = new double[this.rows][this.columns];
        for (int x = 0; x < this.rows; ++x) {
            for (int y = 0; y < this.columns; ++y) {
                newData[x][y] = StrictMath.abs(this.data[x][y]);
            }
        }
        return new Matrix(newData);
    }

    public double sum() {
        double sum = 0.0;
        for (int x = 0; x < this.rows; ++x) {
            for (int y = x; y < this.columns; ++y) {
                sum += this.data[x][y];
            }
        }
        return sum;
    }

    public Matrix getSubMatrix(int startRow, int endRow, int[] selectedColumns) throws MatrixIndexException {
        Matrix subMatrix = new Matrix(endRow - startRow + 1, selectedColumns.length);
        double[][] subMatrixData = subMatrix.getDataRef();
        try {
            for (int i = startRow; i <= endRow; ++i) {
                for (int j = 0; j < selectedColumns.length; ++j) {
                    subMatrixData[i - startRow][j] = this.data[i][selectedColumns[j]];
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new MatrixIndexException("matrix dimension mismatch");
        }
        return subMatrix;
    }

    public Matrix getSubMatrix(int[] selectedRows, int startColumn, int endColumn) throws MatrixIndexException {
        Matrix subMatrix = new Matrix(selectedRows.length, endColumn - startColumn + 1);
        double[][] subMatrixData = subMatrix.getDataRef();
        try {
            for (int i = 0; i < selectedRows.length; ++i) {
                for (int j = startColumn; j <= endColumn; ++j) {
                    subMatrixData[i][j - startColumn] = this.data[selectedRows[i]][j];
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            e.printStackTrace();
            throw new MatrixIndexException("matrix dimension mismatch");
        }
        return subMatrix;
    }

    public String dimString() {
        return "" + this.rows + "rows by " + this.columns + " columns";
    }

    public String toSquareString() {
        return this.toSquareString("\t");
    }

    public String toSquareString(String delim) {
        StringBuilder res = new StringBuilder();
        for (int i = 0; i < this.rows; ++i) {
            res.append(this.data[i][0]);
            for (int j = 1; j < this.columns; ++j) {
                res.append(delim);
                res.append(this.data[i][j]);
            }
            res.append("\n");
        }
        return res.toString();
    }

    public String toLinearString() {
        return this.toLinearString("\t");
    }

    public String toLinearString(String delim) {
        StringBuilder res = new StringBuilder();
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.columns; ++j) {
                if (i + j != 0) {
                    res.append(delim);
                }
                res.append(this.data[i][j]);
            }
        }
        return res.toString();
    }

    public String toString() {
        StringBuilder res = new StringBuilder();
        res.append("Matrix{");
        for (int i = 0; i < this.rows; ++i) {
            if (i > 0) {
                res.append(",");
            }
            res.append("{");
            for (int j = 0; j < this.columns; ++j) {
                if (j > 0) {
                    res.append(",");
                }
                res.append(this.data[i][j]);
            }
            res.append("}");
        }
        res.append("}");
        return res.toString();
    }

    public static Matrix getIdentity(int dimension) {
        Matrix out = new Matrix(dimension, dimension);
        double[][] d = out.getDataRef();
        for (int row = 0; row < dimension; ++row) {
            for (int col = 0; col < dimension; ++col) {
                d[row][col] = row == col ? 1.0 : 0.0;
            }
        }
        return out;
    }

    protected int[] getPermutation() {
        int[] out = new int[this.permutation.length];
        System.arraycopy(this.permutation, 0, out, 0, this.permutation.length);
        return out;
    }

    protected double[][] copyOut() {
        double[][] out = new double[this.rows][this.columns];
        for (int i = 0; i < this.rows; ++i) {
            System.arraycopy(this.data[i], 0, out[i], 0, this.columns);
        }
        return out;
    }

    protected void copyIn(double[][] in) {
        this.rows = in.length;
        this.columns = in[0].length;
        this.data = new double[this.rows][this.columns];
        for (int x = 0; x < this.rows; ++x) {
            System.arraycopy(in[x], 0, this.data[x], 0, this.columns);
        }
        this.lu = null;
    }

    private boolean isValidCoordinate(int row, int col) {
        return row >= 0 && row <= this.rows - 1 && col >= 0 && col <= this.columns - 1;
    }

    public double[] getColumnPackedCopy() {
        double[] vals = new double[this.rows * this.columns];
        for (int i = 0; i < this.rows; ++i) {
            for (int j = 0; j < this.columns; ++j) {
                vals[i + j * this.rows] = this.data[i][j];
            }
        }
        return vals;
    }

    public double[] getRowPackedCopy() {
        double[] vals = new double[this.rows * this.columns];
        for (int i = 0; i < this.rows; ++i) {
            System.arraycopy(this.data[i], 0, vals, i * this.columns, this.columns);
        }
        return vals;
    }

    public static Matrix createRowMatrix(double[] rowData) {
        int nCols = rowData.length;
        double[][] data = new double[1][nCols];
        System.arraycopy(rowData, 0, data[0], 0, nCols);
        return new Matrix(data);
    }

    public static Matrix createColumnMatrix(double[] columnData) {
        int nRows = columnData.length;
        double[][] data = new double[nRows][1];
        for (int row = 0; row < nRows; ++row) {
            data[row][0] = columnData[row];
        }
        return new Matrix(data);
    }
}

